/* VideoPlayer - an application wrapper for the Cortado applet
 * Copyright (C) 2010 Pygraphics, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

package com.fluendo.player;

import java.awt.*;
import java.awt.event.*;
import java.io.*;

import javax.swing.*;

import com.borland.jbcl.layout.*;

public class VideoPlayer extends JFrame {
  private VideoPlayer videoPlayer = this;
  private Cortado applet = null;
  private int locationX, locationY;
  private int playerHeight, playerWidth;
  private int videoHeight, videoWidth;
  private int frameWidth;
  private int growEffectPct = 0;
  private boolean growing;
  private long workOrderTimeStamp = -1;
  private String workOrderPath = null;

  private JPanel mainPanel = new JPanel();
  private OverlayLayout2 overlayLayout21 = new OverlayLayout2();
  private JPanel glassPane = new JPanel();
  private JPanel videoPane = new JPanel();
  private BorderLayout borderLayout1 = new BorderLayout();
  private FrameComponent lblTop = new FrameComponent(this);
  private FrameComponent lblBottom = new FrameComponent(this);
  private FrameComponent lblLeft = new FrameComponent(this);
  private FrameComponent lblRight = new FrameComponent(this);
  private FrameComponent lblTopRight = new FrameComponent(this);
  private FrameComponent lblTopLeft = new FrameComponent(this);
  private FrameComponent lblBottomRight = new FrameComponent(this);
  private FrameComponent lblBottomLeft = new FrameComponent(this);
  private JPanel glassTop = new JPanel();
  private BorderLayout borderLayout2 = new BorderLayout();
  private JPanel glassBottom = new JPanel();
  private BorderLayout borderLayout3 = new BorderLayout();
  private JPanel glassLeft = new JPanel();
  private BorderLayout borderLayout4 = new BorderLayout();
  private JPanel glassRight = new JPanel();
  private BorderLayout borderLayout5 = new BorderLayout();

  private Timer workOrderMonitor = new Timer(500,new ActionListener() {
    public void actionPerformed(ActionEvent e) {
      File file = new File(workOrderPath);
      if (!file.exists())
        shutDown();
      else if (file.lastModified()!=workOrderTimeStamp)
        setVisible(false);
    }
  });

  private Timer growEffectTimer = new Timer(40,new ActionListener() {
    public void actionPerformed(ActionEvent e) {
      growEffectPct = growEffectPct + 4;
      int pct = growEffectPct;
      if (!growing)
        pct = 100 - pct;
      setSize(new Dimension(playerWidth,(playerHeight * pct) / 100));
      if (growEffectPct==100) {
        growEffectTimer.stop();
        growEffectPct = 0;
        if (!growing)
          shutDown();
        else
          startVideo.start();
      }
    }
  });

  private Timer startVideo = new Timer(200,new ActionListener() {
    public void actionPerformed(ActionEvent e) {
      videoPane.add(applet);
      applet.init();
      applet.start();
      workOrderMonitor.start();
    }
  });

  private class FrameComponent extends JLabel {
    private Point startDragPoint;
    private Dimension startSize;
    private Window owner;
    private Icon rolloverIcon = null;
    private Icon icon = null;
    public FrameComponent(Window owner) {
      this.owner = owner;
      addComponentListener(new ComponentAdapter() {
        public void componentResized(ComponentEvent e) {
          resize();
        }
      });
      addMouseListener(new MouseAdapter() {
        public void mousePressed(MouseEvent e) {
          doMousePressed(e);
        }
        public void mouseEntered(MouseEvent e) {
          if (rolloverIcon!=null)
            doSetIcon(rolloverIcon);
        }
        public void mouseExited(MouseEvent e) {
          if (rolloverIcon!=null)
            doSetIcon(icon);
        }
      });
      addMouseMotionListener(new MouseMotionAdapter() {
        public void mouseDragged(MouseEvent e) {
          doMouseDragged(e);
        }
      });

    }
    private void doMousePressed(MouseEvent e) {
      startDragPoint = e.getPoint();
      startSize = owner.getSize();
      if (this==lblTopRight)
        videoPlayer.setVisible(false);
    }
    private void doMouseDragged(MouseEvent e) {
      if (this==lblBottomRight) {
        // resize here
        int width =  startSize.width + e.getX() - startDragPoint.x;
        int height =  startSize.height + e.getY() - startDragPoint.y;
        owner.setSize(new Dimension(width,height));
      } else {
        Point newPanelLocation = owner.getLocation();
        newPanelLocation.translate(e.getX() - startDragPoint.x,e.getY() - startDragPoint.y);
        owner.setLocation(newPanelLocation);
      }
    }
    private void resize() {
      int wLbl = getWidth();
      int hLbl = getHeight();
      if ((wLbl<=0) | (hLbl<=0))
        return;
      ImageIcon imageIcon = (ImageIcon) getIcon();
      if (imageIcon==null)
        return;
      int wImage = imageIcon.getIconWidth();
      int hImage = imageIcon.getIconHeight();
      if ((wLbl==wImage) && (hLbl==hImage))
        return;
      Image newImage = imageIcon.getImage().getScaledInstance(wLbl,hLbl,1);
      setIcon(new ImageIcon(newImage));
    }
    private void doSetIcon(Icon icon) {
      super.setIcon(icon);
    }
    public void setIcon(Icon icon) {
      this.icon = icon;
      doSetIcon(icon);
    }
    public void setRolloverIcon(Icon icon) {
      this.rolloverIcon = icon;
    }
  }

  public VideoPlayer() {
    try {
      System.out.println("Starting player...  (Java:" + System.getProperty("java.version")+")");
      applet = new Cortado();
      readWorkOrder();
      jbInit();
      applet.setBounds(new Rectangle(frameWidth, frameWidth, videoWidth, videoHeight));
      applet.setStub (new CortadoAppletStub (new String[0], applet));
      addWindowListener (new WindowAdapter() {
        public void windowClosing (WindowEvent event) {
          shutDown();
        }
      });
      decorate();
      growEffectTimer.setRepeats(true);
      startVideo.setRepeats(false);
      workOrderMonitor.setRepeats(true);
    } catch (Exception ex) {
      ex.printStackTrace();
    }
    setVisible(true);
  }

  private void jbInit() throws Exception {
    setUndecorated(true);
    setAlwaysOnTop(true);

//    com.sun.awt.AWTUtilities.setWindowOpaque(this, false);

    glassTop.setLayout(borderLayout2);
    glassTop.setOpaque(false);
    glassTop.add(lblTop, java.awt.BorderLayout.CENTER);
    glassTop.add(lblTopRight, java.awt.BorderLayout.EAST);
    glassTop.add(lblTopLeft, java.awt.BorderLayout.WEST);

    glassBottom.setLayout(borderLayout3);
    glassBottom.setOpaque(false);
    glassBottom.add(lblBottom, java.awt.BorderLayout.CENTER);
    glassBottom.add(lblBottomLeft, java.awt.BorderLayout.WEST);
    glassBottom.add(lblBottomRight, java.awt.BorderLayout.EAST);

    glassLeft.setLayout(borderLayout4);
    glassLeft.setOpaque(false);
    glassLeft.add(lblLeft, java.awt.BorderLayout.CENTER);

    glassRight.setLayout(borderLayout5);
    glassRight.setOpaque(false);
    glassRight.add(lblRight, java.awt.BorderLayout.CENTER);

    glassPane.setOpaque(false);
    glassPane.setLayout(borderLayout1);
    glassPane.add(glassBottom, java.awt.BorderLayout.SOUTH);
    glassPane.add(glassRight, java.awt.BorderLayout.EAST);
    glassPane.add(glassTop, java.awt.BorderLayout.NORTH);
    glassPane.add(glassLeft, java.awt.BorderLayout.WEST);

    videoPane.setOpaque(false);
    videoPane.setLayout(null);

    mainPanel.setBackground(Color.black);
    mainPanel.setLayout(overlayLayout21);
    mainPanel.add(videoPane);
    mainPanel.add(glassPane);

    this.getContentPane().add(mainPanel, java.awt.BorderLayout.CENTER);
  }

  private boolean isMacintosh() {
    return (System.getProperty("os.name").toLowerCase().indexOf("mac")!=-1);
  }

  private boolean fileExists(String path) {
    return (new File(path)).exists();
  }

  private String workingDirPath() {
    return System.getProperty("user.dir",".") + '/';
  }

  private String slashifyRel(String path, boolean isDirectory) {
    String p = path;
    if (File.separatorChar != '/')
        p = p.replace(File.separatorChar, '/');
    if (!p.endsWith("/") && isDirectory)
        p = p + "/";
    return p;
  }

  private String pathToProgramDataFolders() {
    String programDataPath;
    // Macontosh Locations
    if (isMacintosh())
      programDataPath = "/Library/Application Support/";
    // Vista Locations
    else if (fileExists("/ProgramData"))
      programDataPath = "/ProgramData/";
    // XP Locations
    else if (fileExists("/Documents and Settings/All Users"))
      programDataPath = "/Documents and Settings/All Users/";
    // Default Locations
    else
      programDataPath = "";
    return slashifyRel(programDataPath, true);
  }

  private String getWorkOrderPath() {
    String appPath = workingDirPath();
    String binName = new File(appPath).getParent();
    String appName = new File(binName).getName();
    String workOrderPath = pathToProgramDataFolders() + appName + "/bin/video_player.ini";
    return workOrderPath;
  }

  private void readWorkOrder() throws IOException {
    workOrderPath = getWorkOrderPath();
    File file = new File(workOrderPath);
    if (!file.exists()) {
      workOrderPath = "video_player.ini"; // this location used when testing
      file = new File(workOrderPath);
      if (!file.exists())
        JOptionPane.showMessageDialog(null,
           "Unable to find video_player.ini.\n\nLocation: "+getWorkOrderPath());
    }
    workOrderTimeStamp = file.lastModified();
    FileReader fileReader = new FileReader(file);
    BufferedReader reader = new BufferedReader(fileReader);
    String line;
    boolean inSection = false;
    while ((line = reader.readLine()) != null) {
      if (line.equalsIgnoreCase("[Params]"))
        inSection = true;
      else if (line.startsWith("["))
        inSection = false;
      else if (line.startsWith("//")) {} // skip
      else if (inSection) {
          int pos = line.indexOf("=");
          if (pos!=-1) {
            String key = line.substring(0,pos);
            String value = line.substring(pos+1);
            applet.setParam(key,value);
          }

      }
    }
    reader.close();
    // collect information for Frames GUI
    frameWidth = applet.getIntParam("frameThickness",30);
    videoWidth = applet.getIntParam("videoWidth",640);
    videoHeight = applet.getIntParam("videoHeight",360);
    playerWidth = videoWidth+(frameWidth*2);
    playerHeight = videoHeight+(frameWidth*2);
    locationX = applet.getIntParam("locationX",0);
    locationY = applet.getIntParam("locationY",0);
  }

  private ImageIcon loadImage(String fullPath) {
    try {
      Image image = getToolkit().getImage(fullPath);
      MediaTracker tracker = new MediaTracker(this);
      tracker.addImage(image, 0);
      tracker.waitForID(0);
      return new ImageIcon(image);
    } catch (Exception e) {
      JOptionPane.showMessageDialog(null,"Problem loading image: "+fullPath);
      return null;
    }
  }

  private void decorate() {
    String skinPath = applet.getParam("skinPath",null);
    lblTop.setIcon(loadImage(skinPath+"video.top.png"));
    lblBottom.setIcon(loadImage(skinPath+"video.bottom.png"));
    lblLeft.setIcon(loadImage(skinPath+"video.left.png"));
    lblRight.setIcon(loadImage(skinPath+"video.right.png"));
    lblTopLeft.setIcon(loadImage(skinPath+"video.topLeft.png"));
    lblTopRight.setIcon(loadImage(skinPath+"video.topRight.png"));
    lblBottomLeft.setIcon(loadImage(skinPath+"video.bottomLeft.png"));
    lblBottomRight.setIcon(loadImage(skinPath+"video.bottomRight.png"));
    lblTopRight.setRolloverIcon(loadImage(skinPath+"video.topRight.rollover.png"));
    lblBottomRight.setRolloverIcon(loadImage(skinPath+"video.bottomRight.rollover.png"));
  }

  public static void main(String args[]) {
    VideoPlayer p;
    p = new VideoPlayer();
    synchronized (p) {
      try {
        p.wait ();
      } catch (InterruptedException ie) {}
    }
  }

  public void setVisible(boolean visible) {
    if (visible==isVisible())
      return;
    if (visible) {
      if (isMacintosh()) {
        setBounds(locationX, locationY, playerWidth, playerHeight);
        super.setVisible(true);
        startVideo.start();
      } else {
        setBounds(locationX, locationY, playerWidth, 0);
        super.setVisible(true);
        growing = true;
        growEffectTimer.start();
      }
    } else {
      if (isMacintosh())
        shutDown();
      applet.pause();
      growing = false;
      growEffectTimer.start();
    }
  }

  public void shutDown() {
    if (applet!=null) {
      applet.stop();
      applet.destroy();
    }
    System.exit(0);
  }

}
